# ArchitectOS (AOS): A Human-in-the-Loop Operating System for LLM‑Driven Software Development

**Author:** Lathem Gibson (Grand Junction, CO, USA)  
**Collaborator:** GPT‑5.1 Thinking (design assistant)  
**Version:** 0.1.0  
**Status:** Draft / Prior Art

---

## Abstract

Large Language Models (LLMs) are already writing significant amounts of production
code, but the way most people use them is fundamentally ad‑hoc: copy‑paste a
snippet from chat into a live repo, hope it works, then debug manually. This
workflow does not scale, is hard to audit, and turns the human developer into a
low‑bandwidth clipboard between an AI and a codebase.

ArchitectOS (AOS) proposes a different approach: treat the LLM as a **planning
and reasoning engine**, and give it a small, deterministic "operating system"
for software projects. The human and LLM collaborate on a **project
specification** that describes:

- which files the model is allowed to touch;
- which tools and commands exist in the environment;
- how to run builds, tests, and sanity checks; and
- how to interpret success and failure.

The AOS CLI (`aos`) then acts as the model's **hands**, executing only the
actions described in the spec, feeding structured results back to the model,
and keeping the entire process reproducible and auditable.

This document defines the motivation, core concepts, system architecture, and
initial reference implementation for ArchitectOS v0.1.0, with an eye toward
establishing clear prior art for a generic LLM‑oriented development OS.

---

## 1. Motivation

Current LLM‑assisted development workflows suffer from several problems:

1. **Copy‑paste anarchy.**  
   The model emits code, the human pastes it into a random editor window, and
   nobody remembers which changes came from where or why.

2. **No shared mental model of the project.**  
   The model sees only the snippets the user pastes in. It has no reliable
   sense of file layout, build tools, or test suites, so it guesses.

3. **Opaque execution.**  
   When code fails, the human runs tests manually, then pastes error messages
   back into the model. There is no structured interface between "plan" and
   "act".

4. **Security and safety concerns.**  
   Letting a model suggest arbitrary shell commands is risky. Humans become
   de‑facto sandboxing systems, which is unsustainable.

5. **Terrible ergonomics for legacy systems.**  
   The projects most in need of help—legacy PHP, old Java monoliths, mystery
   C++—are often the hardest for a model to understand without a lot of manual
   context sharing.

ArchitectOS aims to address these issues by giving both human and model a
**shared, explicit description** of the project and a constrained set of
actions the model can request.

---

## 2. Design Goals

ArchitectOS is built around a few simple but strong goals:

1. **Human‑in‑the‑loop by design.**  
   The human remains the "chief engineer". AOS is a tool, and the LLM is a
   collaborator, not an autonomous agent given root access.

2. **Deterministic, inspectable execution.**  
   All actions go through a small CLI (`aos`) and a simple spec (`project.yml`).
   You can read, diff, and audit both.

3. **Model‑friendly representation.**  
   The spec is YAML/JSON: easy for humans to edit and easy for models to parse
   and reason about.

4. **Tool‑agnostic and stack‑agnostic.**  
   AOS does not try to replace existing tools. It orchestrates them. PHP, Python,
   Node, Make, Docker, Git—anything you can run from a shell can be wired in.

5. **Safe by default.**  
   AOS tries to bias toward read‑only operations first, then controlled write
   operations, always within a small, explicit allowlist.

---

## 3. Core Concepts

### 3.1 Project Spec (`project.yml`)

Every AOS project is defined by a `project.yml` file. At minimum it includes:

- **Metadata**: `name`, `id`, `version`, `description`.
- **Paths**:
  - `root`: canonical path to the project on disk.
  - `scratch`: a working directory for generated artifacts.
  - `include` / `exclude`: glob patterns for files the model may touch.
- **Tools**: named commands the environment supports.
  - Example: `php_lint` → `php -l`.
  - Example: `pytest` → `pytest -q`.
- **Pipelines**: named sequences of steps.
  - Example: `sanity` → "run PHP lint on all allowed PHP files".
  - Example: `test` → "run pytest, then summarize results".

This spec is the "contract" between human, model, and runtime.

### 3.2 The `aos` CLI

The CLI is intentionally small. For v0.1.0 it supports:

- `aos help` — show commands and the philosophy.
- `aos spec` — load and pretty‑print a project spec (or several).
- `aos validate <project-dir>` — run a named pipeline (default: `sanity`).
- `aos plan <project-dir>` — emit a machine‑readable JSON description of the
  project (paths, tools, pipelines) that the LLM can ingest.
- `aos run <project-dir> <step>` — execute a single pipeline step.

The CLI is written in Bash (front) and Python (core), to make it portable and
easy to hack on.

### 3.3 The LLM Interface

ArchitectOS assumes some front‑end that talks to an LLM (ChatGPT, aichat, a
custom UI). That front‑end is responsible for:

- telling the model about the existence of AOS and its capabilities;
- sharing the `project.yml` content and `aos plan` output;
- executing `aos validate` / `aos run` on demand; and
- feeding back logs and JSON results.

In other words: **AOS is the runtime, not the model**.

---

## 4. Architecture (v0.1.0)

The reference implementation in this bundle is intentionally small, but it
embodies the key pieces:

```text
aos (bash)
  └─> aoslib.core (Python)
        ├─ load_project_spec(path)
        ├─ validate_spec(spec)
        ├─ run_pipeline(spec, name)
        └─ run_tool(spec, tool_name, files)
```

### 4.1 Components

- **`aos`**: thin Bash script that:
  - finds its own directory;
  - activates Python; and
  - dispatches to `aoslib.core.main()` with `"$@"`.

- **`aoslib.project`**:
  - defines a `ProjectSpec` class;
  - loads and validates YAML;
  - expands file globs for particular tools.

- **`aoslib.core`**:
  - implements the CLI commands;
  - runs external tools via `subprocess`;
  - captures stdout/stderr and exit codes;
  - emits both human‑readable and JSON summaries.

- **`aoslib.util`**:
  - small helpers: colored output, path utilities, JSON dumping.

- **`examples/wp_rescue/project.yml`**:
  - shows how you might configure AOS to support a WordPress rescue toolkit
    (the original motivating use case).

---

## 5. Example Workflow

Here is what a human + LLM + AOS collaboration might look like in practice.

1. **Initialize the project spec.**

   The human writes an initial `project.yml` describing a legacy WordPress site
   and a new `rescuewp/` toolkit.

2. **Let the model refine the spec.**

   The human pastes `project.yml` into chat and says: _"You are using AOS. Here
   is the current spec. Propose improvements to the tools and pipelines."_  

   The model adds:
   - a `phpunit` tool;
   - a `composer-validate` tool;
   - a `sanity` pipeline that runs both.

3. **Run sanity checks.**

   The human saves the updated `project.yml` and runs:

   ```bash
   ./aos validate examples/wp_rescue
   ```

   AOS runs `php -l` (and any other configured tools) and returns structured
   results. The human pastes relevant logs back to the model.

4. **Iterate on code changes.**

   The model suggests edits to `rescuewp/` PHP files, always staying within the
   `include` paths. The human applies them manually or via a sidecar tool, then
   re‑runs `aos validate`.

5. **Commit and deploy.**

   Once tests and sanity checks pass, the human uses normal Git & deployment
   tooling. AOS stays out of the way.

In later versions, some of these steps can be automated further (e.g. the model
emits `!AOS_RUN sanity` and a small helper script executes it), but the **core
idea remains the same**: the model reasons, AOS acts, the human owns.

---

## 6. Safety and Guardrails

ArchitectOS is not a security product, but it tries not to be reckless:

- All commands are defined in `project.yml`. No free‑form `eval`.
- You can restrict tools to particular file globs.
- Pipelines are named and versioned.
- Nothing touches Git unless you explicitly add it as a tool.
- You can run AOS inside a container or VM for additional isolation.

When in doubt, AOS should bias toward telling you **what it would do** (plan)
before actually doing it (run).

---

## 7. Use Cases

AOS is deliberately generic. Some obvious use cases:

- **WordPress rescue kits** (the original motivation).
- **Legacy PHP / Java / C# monoliths** where nobody alive remembers the build.
- **IoT / embedded projects** (e.g. irrigation controllers on a Raspberry Pi).
- **Data pipelines** where the model needs to orchestrate Python/R scripts.
- **Infrastructure‑as‑Code** refactors (Terraform, Ansible, etc.).

Anywhere you currently find yourself copy‑pasting commands from an LLM into a
terminal, AOS might give you a safer, more repeatable alternative.

---

## 8. Status and Future Work

This bundle represents **ArchitectOS v0.1.0**:

- A minimal repository skeleton.
- A rough but working CLI.
- A concrete example spec.
- Documentation intended to establish prior art.

Short‑term priorities:

1. Stabilize the `project.yml` schema.
2. Define a simple JSON protocol for front‑ends.
3. Add more example templates for common tech stacks.
4. Experiment with lightweight sandboxing.

---

## 9. Conclusion

ArchitectOS is an attempt to give structure to something developers are already
doing informally: using LLMs as co‑pilots for messy real‑world codebases.

By giving both humans and models a **shared OS for software projects**—with an
explicit spec, clear tools, and repeatable pipelines—we can:

- reduce copy‑paste chaos;
- make AI‑assisted work auditable and reproducible;
- and keep the human engineer firmly in control.

This document and the accompanying code are a first, small step toward that
goal.
