import argparse
import subprocess
import sys
from pathlib import Path
from typing import List

from .util import eprint, green, red, yellow, dump_json, find_project_yaml
from .project import ProjectSpec, ToolConfig, PipelineStep


def load_spec_from_dir(project_dir: Path) -> ProjectSpec:
    yml = find_project_yaml(project_dir)
    return ProjectSpec.from_yaml(yml)


def cmd_help(args):
    print("""ArchitectOS (AOS) — v0.1.0

Usage:
  aos help
  aos spec [project-dir...]
  aos validate <project-dir>
  aos plan <project-dir>
  aos run <project-dir> <pipeline-name>

This CLI is intentionally small. It is meant to be driven by a human
and/or an LLM front-end, not to be an agent by itself.
""")


def cmd_spec(args):
    if not args.project_dirs:
        print(yellow("No project directories specified."))
        print("Usage: aos spec <project-dir> [...]")
        return

    for p in args.project_dirs:
        project_dir = Path(p)
        print(green(f"== Project: {project_dir} =="))
        spec = load_spec_from_dir(project_dir)
        print(dump_json(spec.to_dict()))
        print()


def _run_tool(tool: ToolConfig, project_dir: Path, files: List[Path]) -> int:
    cmd = [tool.cmd] + tool.args
    # For v0.1.0 we just append filenames at the end
    cmd += [str(f) for f in files]
    eprint(yellow(f"Running tool {tool.name}: {' '.join(cmd)}"))
    proc = subprocess.run(cmd, cwd=str(project_dir), text=True)
    return proc.returncode


def _expand_file_globs(project_dir: Path, globs: List[str]) -> List[Path]:
    results: List[Path] = []
    for pattern in globs:
        results.extend(project_dir.glob(pattern))
    return sorted(set(results))


def cmd_validate(args):
    project_dir = Path(args.project_dir)
    spec = load_spec_from_dir(project_dir)

    pipeline_name = "sanity"
    if pipeline_name not in spec.pipelines:
        print(red(f"Pipeline '{pipeline_name}' not defined in spec."))
        sys.exit(1)

    steps = spec.pipelines[pipeline_name]
    overall_rc = 0

    for step in steps:
        tool = spec.tools.get(step.tool)
        if not tool:
            print(red(f"Tool '{step.tool}' not defined for step '{step.step}'."))
            overall_rc = 1
            continue

        files = _expand_file_globs(project_dir, tool.file_globs)
        if not files:
            eprint(yellow(f"[{step.step}] No files matched for tool '{tool.name}'."))
            continue

        rc = _run_tool(tool, project_dir, files)
        if rc != 0:
            overall_rc = rc
            eprint(red(f"[{step.step}] FAILED with exit code {rc}"))
        else:
            eprint(green(f"[{step.step}] OK"))

    sys.exit(overall_rc)


def cmd_plan(args):
    project_dir = Path(args.project_dir)
    spec = load_spec_from_dir(project_dir)
    # Emit a model-friendly JSON summary
    summary = {
        "project_dir": str(project_dir),
        "spec": spec.to_dict(),
    }
    print(dump_json(summary))


def cmd_run(args):
    project_dir = Path(args.project_dir)
    pipeline_name = args.pipeline
    spec = load_spec_from_dir(project_dir)

    if pipeline_name not in spec.pipelines:
        print(red(f"Pipeline '{pipeline_name}' not defined in spec."))
        sys.exit(1)

    steps = spec.pipelines[pipeline_name]
    overall_rc = 0

    for step in steps:
        tool = spec.tools.get(step.tool)
        if not tool:
            print(red(f"Tool '{step.tool}' not defined for step '{step.step}'."))
            overall_rc = 1
            continue

        files = _expand_file_globs(project_dir, tool.file_globs)
        rc = _run_tool(tool, project_dir, files)
        if rc != 0:
            overall_rc = rc
            eprint(red(f"[{step.step}] FAILED with exit code {rc}"))
        else:
            eprint(green(f"[{step.step}] OK"))

    sys.exit(overall_rc)


def main(argv=None):
    parser = argparse.ArgumentParser(prog="aos", add_help=False)
    subparsers = parser.add_subparsers(dest="command")

    sp_help = subparsers.add_parser("help")
    sp_help.set_defaults(func=cmd_help)

    sp_spec = subparsers.add_parser("spec")
    sp_spec.add_argument("project_dirs", nargs="*")
    sp_spec.set_defaults(func=cmd_spec)

    sp_validate = subparsers.add_parser("validate")
    sp_validate.add_argument("project_dir")
    sp_validate.set_defaults(func=cmd_validate)

    sp_plan = subparsers.add_parser("plan")
    sp_plan.add_argument("project_dir")
    sp_plan.set_defaults(func=cmd_plan)

    sp_run = subparsers.add_parser("run")
    sp_run.add_argument("project_dir")
    sp_run.add_argument("pipeline")
    sp_run.set_defaults(func=cmd_run)

    args, unknown = parser.parse_known_args(argv)

    if not getattr(args, "command", None):
        cmd_help(args)
        return

    args.func(args)


if __name__ == "__main__":
    main()
