from dataclasses import dataclass, field
from pathlib import Path
from typing import Dict, List, Any

from .util import load_yaml


@dataclass
class ToolConfig:
    name: str
    cmd: str
    args: List[str] = field(default_factory=list)
    file_globs: List[str] = field(default_factory=list)


@dataclass
class PipelineStep:
    step: str
    tool: str
    files_from: str


@dataclass
class ProjectSpec:
    name: str
    id: str
    version: str
    description: str = ""
    root: Path = Path(".")
    scratch: Path = Path(".aos-scratch")
    include: List[str] = field(default_factory=list)
    exclude: List[str] = field(default_factory=list)
    tools: Dict[str, ToolConfig] = field(default_factory=dict)
    pipelines: Dict[str, List[PipelineStep]] = field(default_factory=dict)

    @classmethod
    def from_yaml(cls, path: Path) -> "ProjectSpec":
        data = load_yaml(path)
        meta = {
            "name": data.get("name", ""),
            "id": data.get("id", ""),
            "version": data.get("version", ""),
            "description": data.get("description", ""),
        }
        paths = data.get("paths", {}) or {}
        tools_data = data.get("tools", {}) or {}
        pipelines_data = data.get("pipelines", {}) or {}

        tools = {
            name: ToolConfig(
                name=name,
                cmd=cfg.get("cmd", ""),
                args=list(cfg.get("args", []) or []),
                file_globs=list(cfg.get("file_globs", []) or []),
            )
            for name, cfg in tools_data.items()
        }

        pipelines: Dict[str, List[PipelineStep]] = {}
        for pname, steps in pipelines_data.items():
            pipelines[pname] = [
                PipelineStep(
                    step=step.get("step", ""),
                    tool=step.get("tool", ""),
                    files_from=step.get("files_from", ""),
                )
                for step in (steps or [])
            ]

        return cls(
            root=Path(paths.get("root", ".")),
            scratch=Path(paths.get("scratch", ".aos-scratch")),
            include=list(paths.get("include", []) or []),
            exclude=list(paths.get("exclude", []) or []),
            tools=tools,
            pipelines=pipelines,
            **meta,
        )

    def to_dict(self) -> Dict[str, Any]:
        return {
            "name": self.name,
            "id": self.id,
            "version": self.version,
            "description": self.description,
            "paths": {
                "root": str(self.root),
                "scratch": str(self.scratch),
                "include": self.include,
                "exclude": self.exclude,
            },
            "tools": {
                name: {
                    "cmd": tool.cmd,
                    "args": tool.args,
                    "file_globs": tool.file_globs,
                }
                for name, tool in self.tools.items()
            },
            "pipelines": {
                pname: [
                    {
                        "step": s.step,
                        "tool": s.tool,
                        "files_from": s.files_from,
                    }
                    for s in steps
                ]
                for pname, steps in self.pipelines.items()
            },
        }
