import json
import os
from dataclasses import dataclass, asdict
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional

TOOLCHAIN_ROOT_ENV = "AOS_TOOLCHAIN_ROOT"


def _default_user_root() -> Path:
    return Path(os.path.expanduser("~/.architectos/toolchains"))


@dataclass
class ToolchainMeta:
    id: str
    language: str
    name: str
    created_at: str
    created_by: str
    version: Optional[str] = None
    base_interpreter: Optional[str] = None
    tags: Optional[List[str]] = None


@dataclass
class ToolchainStatus:
    last_used: Optional[str] = None
    projects: Optional[List[str]] = None
    health: str = "unknown"
    notes: Optional[List[str]] = None


@dataclass
class Toolchain:
    root: Path
    meta: ToolchainMeta
    status: ToolchainStatus


def get_toolchain_root() -> Path:
    root = os.getenv(TOOLCHAIN_ROOT_ENV)
    if root:
        return Path(root).expanduser().resolve()
    return _default_user_root()


def _load_json(path: Path) -> Dict:
    if not path.exists():
        return {}
    with path.open("r", encoding="utf-8") as f:
        return json.load(f)


def _save_json(path: Path, data: Dict) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    tmp = path.with_suffix(".tmp")
    with tmp.open("w", encoding="utf-8") as f:
        json.dump(data, f, indent=2, sort_keys=True)
    tmp.replace(path)


def list_toolchains(language: Optional[str] = None) -> List[Toolchain]:
    root = get_toolchain_root()
    if not root.exists():
        return []
    results: List[Toolchain] = []
    for lang_dir in root.iterdir():
        if not lang_dir.is_dir():
            continue
        lang = lang_dir.name
        if language and lang != language:
            continue
        for tc_dir in lang_dir.iterdir():
            if not tc_dir.is_dir():
                continue
            meta_path = tc_dir / "meta.json"
            status_path = tc_dir / "status.json"
            meta_raw = _load_json(meta_path)
            status_raw = _load_json(status_path)
            if not meta_raw:
                # Skip unknown structures
                continue
            meta = ToolchainMeta(**meta_raw)
            status = ToolchainStatus(**status_raw) if status_raw else ToolchainStatus()
            results.append(Toolchain(root=tc_dir, meta=meta, status=status))
    return results


def create_python_toolchain(name: str, python_exe: Optional[str] = None, tags: Optional[List[str]] = None) -> Toolchain:
    """Skeleton implementation: create metadata for a Python toolchain.

    The actual environment creation (venv + pip install) is left to future
    versions; this function simply reserves a directory and writes meta.json.
    """
    root = get_toolchain_root() / "python" / name
    root.mkdir(parents=True, exist_ok=True)
    tc_id = f"python:{name}"
    now = datetime.utcnow().isoformat() + "Z"
    meta = ToolchainMeta(
        id=tc_id,
        language="python",
        name=name,
        created_at=now,
        created_by="aos toolchain create python",
        version=None,
        base_interpreter=python_exe,
        tags=tags or [],
    )
    status = ToolchainStatus(last_used=None, projects=[], health="empty", notes=["skeleton toolchain"])
    _save_json(root / "meta.json", asdict(meta))
    _save_json(root / "status.json", asdict(status))
    return Toolchain(root=root, meta=meta, status=status)


def bind_toolchain_to_project(language: str, name: str, project_root: Path) -> None:
    """Bind a named toolchain to the given project by writing
    .architectos/toolchain.yml. Existing configuration is updated.
    """
    tc_id = f"{language}:{name}"
    cfg_path = project_root / ".architectos" / "toolchain.yml"
    cfg: Dict[str, Dict[str, str]] = {}
    if cfg_path.exists():
        import yaml  # type: ignore
        with cfg_path.open("r", encoding="utf-8") as f:
            cfg = yaml.safe_load(f) or {}
    lang_cfg = cfg.get(language, {})
    lang_cfg["toolchain_id"] = tc_id
    cfg[language] = lang_cfg

    cfg_path.parent.mkdir(parents=True, exist_ok=True)
    import yaml  # type: ignore
    with cfg_path.open("w", encoding="utf-8") as f:
        yaml.safe_dump(cfg, f)

    # Update status.json for the toolchain, if it exists
    for tc in list_toolchains(language=language):
        if tc.meta.name == name:
            projects = tc.status.projects or []
            proj_str = str(project_root)
            if proj_str not in projects:
                projects.append(proj_str)
            tc.status.projects = projects
            tc.status.last_used = datetime.utcnow().isoformat() + "Z"
            _save_json(tc.root / "status.json", asdict(tc.status))
            break
