import os
import argparse

from .config import load_project_config
from .mode_manager import ModeManager
from . import rag_cli

def main(argv=None) -> None:
    parser = argparse.ArgumentParser(prog="aos", description="ArchitectOS CLI (0.3.0 skeleton)")
    sub = parser.add_subparsers(dest="command", required=True)

    p_info = sub.add_parser("info", help="Show basic project and domain information")
    p_info.set_defaults(command="info")

    p_modes = sub.add_parser("modes", help="Describe modes and their default domains")
    p_modes.set_defaults(command="modes")

    p_rag = sub.add_parser("rag", help="Invoke RAG subcommands (wrapper around aos-rag)")
    p_rag.add_argument("rag_args", nargs=argparse.REMAINDER)
    p_rag.set_defaults(command="rag")

    args = parser.parse_args(argv)
    mm = ModeManager()

    if args.command == "info":
        cfg = load_project_config()
        domain = os.getenv("AOS_DOMAIN", "SAFE")
        print(f"Project: {cfg.get('name', 'unknown')} {cfg.get('version', '')}")
        print(f"Execution domain: {domain}")
    elif args.command == "modes":
        print(mm.describe())
    elif args.command == "rag":
        # Delegate to rag_cli main
        rag_cli.main(args.rag_args)
    else:
        parser.error(f"Unknown command: {args.command!r}")

if __name__ == "__main__":
    main()
