from dataclasses import dataclass
from typing import Dict, Optional

from .config import load_constraints_config
from .domains import build_domains_from_config, Domain

@dataclass
class Mode:
    name: str
    default_domain: str

class ModeManager:
    def __init__(self) -> None:
        cfg = load_constraints_config()
        self.domains: Dict[str, Domain] = build_domains_from_config(cfg)
        self.mode_defaults: Dict[str, str] = {
            name: dom for name, dom in (cfg.get("modes") or {}).items()
        }

    def get_mode(self, name: str) -> Mode:
        default_domain = self.mode_defaults.get(name, "SAFE")
        return Mode(name=name, default_domain=default_domain)

    def get_domain(self, name: str) -> Optional[Domain]:
        return self.domains.get(name)

    def describe(self) -> str:
        lines = []
        lines.append("Execution domains:")
        for d in self.domains.values():
            lines.append(f"- {d.name}: {d.description}")
        lines.append("")
        lines.append("Mode defaults:")
        for mode, dom in self.mode_defaults.items():
            lines.append(f"- {mode} -> {dom}")
        return "\n".join(lines)
