from __future__ import annotations

from dataclasses import dataclass
from typing import Dict, List, Literal, Optional
import os
import json
from pathlib import Path

import requests

from .config import get_project_root, load_yaml as _load_yaml


ProviderName = Literal["claude", "gemini"]


@dataclass
class LLMConfig:
    provider: ProviderName
    model: str
    max_output_tokens: int = 2048
    temperature: float = 0.3
    claude_endpoint: str = "https://api.anthropic.com/v1/messages"
    gemini_endpoint: str = "https://generativelanguage.googleapis.com/v1beta/models"


def _load_llm_config() -> LLMConfig:
    root = get_project_root()
    cfg_path = root / ".architectos" / "llm.yml"
    cfg: Dict = {}

    if cfg_path.exists():
        cfg = _load_yaml(cfg_path)

    provider: ProviderName = (  # type: ignore[assignment]
        os.getenv("AOS_LLM_PROVIDER")
        or cfg.get("provider")
        or "claude"
    )

    model = os.getenv("AOS_LLM_MODEL") or cfg.get("model")
    if not model:
        if provider == "claude":
            model = "claude-3-5-sonnet-20241022"
        elif provider == "gemini":
            model = "gemini-2.5-pro"

    max_output_tokens = int(
        os.getenv("AOS_LLM_MAX_TOKENS", cfg.get("max_output_tokens", 2048))
    )
    temperature = float(
        os.getenv("AOS_LLM_TEMPERATURE", cfg.get("temperature", 0.3))
    )

    endpoints = cfg.get("endpoints", {}) or {}
    return LLMConfig(
        provider=provider,
        model=model,
        max_output_tokens=max_output_tokens,
        temperature=temperature,
        claude_endpoint=endpoints.get(
            "claude", "https://api.anthropic.com/v1/messages"
        ),
        gemini_endpoint=endpoints.get(
            "gemini", "https://generativelanguage.googleapis.com/v1beta/models"
        ),
    )


class LLMClient:
    """Thin abstraction over multiple LLM providers.

    We accept OpenAI-style messages:

        [{"role": "system"|"user"|"assistant", "content": "text"}, ...]

    and map them to each vendor's native format.
    """

    def __init__(self, cfg: Optional[LLMConfig] = None) -> None:
        self.cfg = cfg or _load_llm_config()

    def complete(self, messages: List[Dict[str, str]], **kwargs) -> str:
        provider = self.cfg.provider
        if provider == "claude":
            return self._call_claude(messages, **kwargs)
        if provider == "gemini":
            return self._call_gemini(messages, **kwargs)
        raise ValueError(f"Unsupported provider: {provider!r}")

    # --------- Claude ----------

    def _call_claude(self, messages: List[Dict[str, str]], **kwargs) -> str:
        """Call Claude Messages API via HTTP."""
        api_key = os.getenv("ANTHROPIC_API_KEY")
        if not api_key:
            raise RuntimeError("ANTHROPIC_API_KEY is not set")

        # Anthropic wants role/content pairs where content is an array of blocks.
        claude_messages = []
        system: Optional[str] = None

        for m in messages:
            role = m.get("role", "user")
            content = m.get("content", "")

            if role == "system":
                system = (system + "\n" + content) if system else content
                continue

            claude_messages.append(
                {
                    "role": role,
                    "content": [
                        {"type": "text", "text": content},
                    ],
                }
            )

        body = {
            "model": self.cfg.model,
            "max_output_tokens": kwargs.get(
                "max_output_tokens", self.cfg.max_output_tokens
            ),
            "temperature": kwargs.get("temperature", self.cfg.temperature),
            "messages": claude_messages,
        }
        if system:
            body["system"] = system

        headers = {
            "x-api-key": api_key,
            "anthropic-version": "2023-06-01",
            "content-type": "application/json",
        }

        resp = requests.post(
            self.cfg.claude_endpoint,
            headers=headers,
            data=json.dumps(body),
            timeout=60,
        )
        resp.raise_for_status()
        data = resp.json()

        blocks = data.get("content") or []
        for blk in blocks:
            if blk.get("type") == "text":
                return blk.get("text", "")
        return json.dumps(data)

    # --------- Gemini ----------

    def _call_gemini(self, messages: List[Dict[str, str]], **kwargs) -> str:
        """Call Gemini generateContent via REST."""
        api_key = os.getenv("GEMINI_API_KEY")
        if not api_key:
            raise RuntimeError("GEMINI_API_KEY is not set")

        contents = []
        for m in messages:
            role = m.get("role", "user")
            content = m.get("content", "")
            # Keep it simple: treat system prompts as user role for Gemini.
            contents.append(
                {
                    "role": "user" if role == "system" else role,
                    "parts": [{"text": content}],
                }
            )

        body = {
            "contents": contents,
            "generationConfig": {
                "maxOutputTokens": kwargs.get(
                    "max_output_tokens", self.cfg.max_output_tokens
                ),
                "temperature": kwargs.get("temperature", self.cfg.temperature),
            },
        }

        url = f"{self.cfg.gemini_endpoint}/{self.cfg.model}:generateContent"
        headers = {
            "x-goog-api-key": api_key,
            "content-type": "application/json",
        }

        resp = requests.post(url, headers=headers, data=json.dumps(body), timeout=60)
        resp.raise_for_status()
        data = resp.json()

        candidates = data.get("candidates") or []
        if not candidates:
            return json.dumps(data)
        parts = candidates[0].get("content", {}).get("parts", [])
        texts = [p.get("text", "") for p in parts if "text" in p]
        return "\n".join(t for t in texts if t)
