import shlex
import subprocess
from typing import Iterable

from .config import load_constraints_config
from .domains import build_domains_from_config
from .logging_utils import append_log


def _load_unbound_domain():
    cfg = load_constraints_config()
    domains = build_domains_from_config(cfg)
    return domains.get("UNBOUND")


def confirm_required(cmd: str, patterns: Iterable[str]) -> bool:
    text = cmd.strip()
    for pat in patterns:
        if pat in text:
            return True
    return False


def run_unbound_shell(command: str, auto_confirm: bool = False) -> int:
    """Run a shell command in UNBOUND domain.

    This is a thin skeleton: it logs commands and can enforce confirmation
    prompts on dangerous patterns. The actual UI / UX layer should handle
    user prompts and route the confirmation flag appropriately.
    """
    domain = _load_unbound_domain()
    if not domain:
        raise RuntimeError("UNBOUND domain is not configured")

    needs_confirm = confirm_required(command, domain.confirm_patterns or [])
    if needs_confirm and not auto_confirm:
        raise PermissionError(
            f"Command requires confirmation in UNBOUND domain: {command!r}"
        )

    log_path = domain.log_path or "~/.architectos/unbound/logs/unbound.log"
    append_log(log_path, f"UNBOUND RUN: {command}")

    return subprocess.call(command, shell=True)
