#!/usr/bin/env bash
#
# ArchitectOS 0.3.0 installer (Unix skeleton)
#
# Usage:
#   ./install.sh --user      # default, installs into ~/.local/share/architectos
#   ./install.sh --system    # installs into /opt/architectos (requires sudo)
#
set -euo pipefail

MODE="user"

while [[ $# -gt 0 ]]; do
  case "$1" in
    --user)
      MODE="user"
      shift
      ;;
    --system)
      MODE="system"
      shift
      ;;
    *)
      echo "Unknown option: $1" >&2
      echo "Usage: $0 [--user|--system]" >&2
      exit 1
      ;;
  esac
done

VERSION="0.3.0"
SRC_ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

if [[ "$MODE" == "system" ]]; then
  INSTALL_ROOT="/opt/architectos"
  PREFIX="/usr/local"
else
  INSTALL_ROOT="${HOME}/.local/share/architectos"
  PREFIX="${HOME}/.local"
fi

BIN_DIR="${PREFIX}/bin"
VERSION_DIR="${INSTALL_ROOT}/${VERSION}"
CURRENT_LINK="${INSTALL_ROOT}/current"
VENV_PATH="${INSTALL_ROOT}/venv"

echo "[install] Mode: $MODE"
echo "[install] Install root: $INSTALL_ROOT"
echo "[install] Version dir: $VERSION_DIR"
echo "[install] Bin dir: $BIN_DIR"
echo "[install] Venv: $VENV_PATH"

mkdir -p "$VERSION_DIR"
mkdir -p "$BIN_DIR"

echo "[install] Copying files..."
tar c -C "$SRC_ROOT" . | tar x -C "$VERSION_DIR"

echo "[install] Updating 'current' symlink..."
ln -sfn "$VERSION_DIR" "$CURRENT_LINK"

# Create or update the tool venv
if [[ ! -d "$VENV_PATH" ]]; then
  echo "[install] Creating virtualenv..."
  python3 -m venv "$VENV_PATH"
fi

echo "[install] Installing Python dependencies..."
"$VENV_PATH/bin/pip" install --upgrade pip
if [[ -f "$VERSION_DIR/requirements.txt" ]]; then
  "$VENV_PATH/bin/pip" install -r "$VERSION_DIR/requirements.txt"
fi

# Create wrapper scripts in $BIN_DIR pointing at the installed version
make_wrapper () {
  local name="$1"
  local target="$2"
  local path="${BIN_DIR}/${name}"
  cat > "$path" <<EOF
#!/usr/bin/env bash
set -euo pipefail
INSTALL_ROOT="${INSTALL_ROOT}"
VERSION="${VERSION}"
VENV_PATH="\${INSTALL_ROOT}/venv"
export AOS_INSTALL_ROOT="\${INSTALL_ROOT}"
export AOS_PROJECT_ROOT="\${INSTALL_ROOT}/current"
export VIRTUAL_ENV="\${VENV_PATH}"
export PATH="\${VENV_PATH}/bin:\$PATH"
exec "\${INSTALL_ROOT}/current/${target}" "\$@"
EOF
  chmod +x "$path"
  echo "[install] Installed wrapper: $path -> ${target}"
}

make_wrapper "aos" "bin/aos"
make_wrapper "aos-rag" "bin/aos-rag"
make_wrapper "aos-unbound" "bin/aos-unbound"
make_wrapper "aos-doctor" "bin/aos-doctor"
make_wrapper "aos-upgrade" "bin/aos-upgrade"
make_wrapper "aos-tmux" "bin/aos-tmux"

echo "[install] Done."
echo "  Binaries installed to: $BIN_DIR"
echo "  Try running: 'aos info'"
