import os
import sys
import argparse

from .config import load_project_config, get_project_root
from .mode_manager import ModeManager
from . import rag_cli
from . import toolchain_manager as tcm


FIRST_RUN_FLAG = os.path.expanduser("~/.config/architectos/first_run_done")


def _is_interactive() -> bool:
    try:
        return sys.stdin.isatty()
    except Exception:
        return False


def _mark_first_run_done() -> None:
    path = FIRST_RUN_FLAG
    d = os.path.dirname(path)
    os.makedirs(d, exist_ok=True)
    with open(path, "w", encoding="utf-8") as f:
        f.write("1\n")


def _first_run_needed() -> bool:
    if os.getenv("AOS_NO_WIZARD") == "1":
        return False
    if os.path.exists(FIRST_RUN_FLAG):
        return False
    if tcm.list_toolchains():
        return False
    return True


def maybe_run_first_run_wizard() -> None:
    """Run a small first-run setup wizard if appropriate."""
    if not _first_run_needed():
        return

    if not _is_interactive():
        if os.getenv("AOS_FIRST_RUN_AUTO") == "1":
            tcm.create_default_python_toolchain()
        _mark_first_run_done()
        return

    print("Welcome to ArchitectOS 0.3.0 (first launch)")
    print("")
    print("I can set up a default Python toolchain so ArchitectOS and your projects")
    print("run in a consistent, isolated environment.")
    print("")
    print("Create a default Python toolchain now?")
    print("[1] Yes, create a default toolchain (recommended)")
    print("[2] No, I'll do it later")
    choice = input("> ").strip() or "2"

    if choice == "1":
        tc = tcm.create_default_python_toolchain()
        if tc is None:
            print("Failed to create default Python toolchain; you can try again later with:")
            print("  aos toolchain create python NAME")
        else:
            project_root = get_project_root()
            if (project_root / "project.yml").exists():
                tcm.bind_toolchain_to_project("python", tc.meta.name, project_root)
                print(f"Created and bound default toolchain 'python:{tc.meta.name}' to {project_root}")
            else:
                print(f"Created default toolchain 'python:{tc.meta.name}'.")
    else:
        print("Skipping default toolchain creation. You can create one later with:")
        print("  aos toolchain create python NAME")
        print("  aos toolchain use python NAME")

    _mark_first_run_done()


def _cmd_info(args: argparse.Namespace) -> None:
    cfg = load_project_config()
    domain = os.getenv("AOS_DOMAIN", "SAFE")
    print(f"Project: {cfg.get('name', 'unknown')} {cfg.get('version', '')}")
    print(f"Execution domain: {domain}")


def _cmd_modes(args: argparse.Namespace) -> None:
    mm = ModeManager()
    print(mm.describe())


def _cmd_rag(args: argparse.Namespace) -> None:
    rag_cli.main(args.rag_args)


def _cmd_toolchain(args: argparse.Namespace) -> None:
    if args.tcmd == "list":
        tcs = tcm.list_toolchains(language=args.language)
        if not tcs:
            print("No toolchains found.")
            return
        for tc in tcs:
            status = tc.status
            print(
                f"{tc.meta.language:7} {tc.meta.name:24} {status.health:8} "
                f"{(status.last_used or '-'):20} "
                f"{len(status.projects or [])} project(s)"
            )
    elif args.tcmd == "create":
        if args.language != "python":
            print("Only Python toolchains are supported in this skeleton.")
            return
        tc = tcm.create_python_toolchain(
            args.name,
            python_exe=args.python,
            tags=args.tag,
            create_venv=True,
        )
        print(f"Created Python toolchain at: {tc.root}")
    elif args.tcmd == "use":
        project_root = get_project_root()
        tcm.bind_toolchain_to_project(args.language, args.name, project_root)
        print(f"Bound toolchain {args.language}:{args.name} to project {project_root}")
    else:
        raise SystemExit(f"Unknown toolchain subcommand: {args.tcmd!r}")



def _cmd_llm_test(args):
    from .llm_client import LLMClient
    client = LLMClient()
    try:
        resp = client.complete([
            {"role": "user", "content": "Test message from ArchitectOS."}
        ])
        print("LLM OK:\n", resp)
    except Exception as e:
        print("LLM ERROR:", e)

def main(argv=None) -> None:
    parser = argparse.ArgumentParser(prog="aos", description="ArchitectOS CLI (0.3.0 skeleton)")
    sub = parser.add_subparsers(dest="command", required=True)

    # LLM test
    p_llm = sub.add_parser("llm", help="LLM utilities")
    llm_sub = p_llm.add_subparsers(dest="llmcmd", required=True)
    p_llm_test = llm_sub.add_parser("test", help="Test LLM connectivity")
    p_llm_test.set_defaults(func=_cmd_llm_test)

    p_info = sub.add_parser("info", help="Show basic project and domain information")
    p_info.set_defaults(func=_cmd_info)

    p_modes = sub.add_parser("modes", help="Describe modes and their default domains")
    p_modes.set_defaults(func=_cmd_modes)

    p_rag = sub.add_parser("rag", help="Invoke RAG subcommands (wrapper around aos-rag)")
    p_rag.add_argument("rag_args", nargs=argparse.REMAINDER)
    p_rag.set_defaults(func=_cmd_rag)

    p_tc = sub.add_parser("toolchain", help="Manage language/runtime toolchains")
    tc_sub = p_tc.add_subparsers(dest="tcmd", required=True)

    p_tc_list = tc_sub.add_parser("list", help="List known toolchains")
    p_tc_list.add_argument("--language", help="Filter by language (e.g. python, node)")
    p_tc_list.set_defaults(func=_cmd_toolchain)

    p_tc_create = tc_sub.add_parser("create", help="Create a new toolchain (Python only in skeleton)")
    p_tc_create.add_argument("language", help="Language (python, node, etc.)")
    p_tc_create.add_argument("name", help="Toolchain name")
    p_tc_create.add_argument("--python", help="Path to python interpreter (for python language)")
    p_tc_create.add_argument("--tag", action="append", help="Tag to attach to the toolchain", default=None)
    p_tc_create.set_defaults(func=_cmd_toolchain)

    p_tc_use = tc_sub.add_parser("use", help="Bind a toolchain to the current project")
    p_tc_use.add_argument("language", help="Language (python, node, etc.)")
    p_tc_use.add_argument("name", help="Toolchain name")
    p_tc_use.set_defaults(func=_cmd_toolchain)

    args = parser.parse_args(argv)

    maybe_run_first_run_wizard()
    args.func(args)


if __name__ == "__main__":
    main()
