# ArchitectOS LLM Integration (0.3.5)

This document describes the multi-LLM integration for ArchitectOS 0.3.5.

The goal is to make ArchitectOS independent of any single LLM vendor while
providing a simple, consistent interface from the rest of the system.

## Overview

The module `architectos.llm_client` exposes a single entrypoint:

- `LLMClient.complete(messages: List[dict], **kwargs) -> str`

It accepts messages in an OpenAI-style shape:

```python
messages = [
    {"role": "system", "content": "You are ArchitectOS."},
    {"role": "user", "content": "Explain how this project works."},
]
```

The client then routes these messages to the active provider and returns the
text response.

## Configuration

The LLM configuration is resolved from:

1. `.architectos/llm.yml` (per-project configuration)
2. Environment variables (which override the file)

### Example `.architectos/llm.yml`

```yaml
provider: claude          # or: gemini
model: claude-3-5-sonnet-20241022

max_output_tokens: 2048
temperature: 0.3

endpoints:
  claude: "https://api.anthropic.com/v1/messages"
  gemini: "https://generativelanguage.googleapis.com/v1beta/models"
```

### Environment overrides

- `AOS_LLM_PROVIDER` (e.g. `claude`, `gemini`)
- `AOS_LLM_MODEL` (provider-specific model id)
- `AOS_LLM_MAX_TOKENS` (integer)
- `AOS_LLM_TEMPERATURE` (float)

API keys must be set via environment variables:

- `ANTHROPIC_API_KEY` for Claude
- `GEMINI_API_KEY` for Gemini

## Providers

### Claude

The client uses the Claude Messages HTTP API. System messages are collapsed
into a single `system` string, and user/assistant messages are converted into
role/content block objects as expected by the API.

### Gemini

The client uses the Gemini `generateContent` HTTP endpoint. Messages are
converted into a list of `contents`, with each message mapped to a role and
text parts. System messages are treated as `user` role for simplicity.

## Usage from Python

```python
from architectos.llm_client import LLMClient

client = LLMClient()
resp = client.complete(
    [
        {"role": "system", "content": "You are ArchitectOS."},
        {"role": "user", "content": "Describe the SAFE/NET/UNBOUND domains."},
    ]
)

print(resp)
```

The rest of ArchitectOS can be migrated to call `LLMClient` instead of
hard-coding a particular provider, enabling future support for additional
vendors (including self-hosted or region-specific models).
